function [xo,yo] = bar(arg1,arg2,arg3,arg4,arg5)
%BAR	Bar chart.
%	BAR(Y) draws a bar chart of the elements of vector Y.
%	BAR(X,Y) draws a bar chart of the elements of vector Y at
%	the locations specified in vector X.  The X-values must
%	be in ascending order.  If X and Y are matrices the same size,
%	one bar chart per column is drawn.  The color of the bars is 
%	determined by the colormap.
%
%	BAR(X,Y,OVERLAP,SPACING) or BAR(Y,OVERLAP,SPACING) uses the
%	values OVERLAP and SPACING to set the fractional overlap of the
%	bars in a group and the factional spacing between groups.
%	OVERLAP defaults to -0.15, SPACING defaults to 0.25.
%
%	BAR(...,'overlap') produces the default overlapped bar chart.
%	BAR(...,'stacked') produces a stacked bar chart.
%
%	[XX,YY] = BAR(X,Y) does not draw a graph, but returns vectors
%	X and Y such that PLOT(XX,YY) is a bar chart.
%
%	H = BAR(...) returns a vector of patch handles.
%
%	Example:
%	    subplot(3,1,1), bar(rand(10,5),'stacked')
%	    subplot(3,1,2), bar(0:.25:1,rand(5),0,.25)
%	    subplot(3,1,3), colormap(cool)
%	    h=bar(rand(7,3),.25,.3); set(h,'edgeColor','None')
%
%	See also COLUMN, HIST.

%	C.B Moler 2-06-86
%	Modified 24-Dec-88, 2-Jan-92 LS.
%	Modified 8-5-91, 3-7-94 by cmt;
%	Copyright (c) 1984-94 by The MathWorks, Inc.

defaultoverlap = -.1; % Between bar spacing
defaultspacing = .25; % Between group spacing
stacked = 0; % Assume overlapped bar plot.

if nargin == 1, % bar(y)
  if isstr(arg1)
	error('First argument must be numeric.');
  end
  if min(size(arg1))==1, y = arg1(:); else y = arg1; end
  [n,m] = size(y);
  x = [1:n]'*ones(1,m);
elseif nargin == 2, % bar(x,y) or bar(y,'kind')
  if isstr(arg1)
	error('First argument must be numeric.');
  end
  if isstr(arg2),
    kind = [lower(arg2) '  ']; % Protect against short string
    if all(kind(1:2)=='ov'), % overlap
      stacked = 0;
    elseif all(kind(1:2)=='st'), % stacked
      stacked = 1;
    else
      warning('This linetype syntax is obsolete.  Use BAR(Y) instead.');
    end
    if min(size(arg1))==1, y = arg1(:); else y = arg1; end
    [n,m] = size(y);
    x = [1:n]'*ones(1,m);
  else
    if min(size(arg2))==1, y = arg2(:); else y = arg2; end
    [n,m] = size(y);
    if min(size(arg1))==1, x = arg1(:)*ones(1,m); else x = arg1; end
  end
elseif nargin==3, % bar(x,y,'kind') or bar(y,overlap,spacing)
  if isstr(arg3),
    kind = [lower(arg3) '  ']; % Protect against short string
    if all(kind(1:2)=='ov'), % overlap
      stacked = 0;
    elseif all(kind(1:2)=='st'), % stacked
      stacked = 1;
    else
      warning('This linetype syntax is obsolete.  Use BAR(Y) instead.');
    end
    if min(size(arg2))==1, y = arg2(:); else y = arg2; end
    [n,m] = size(y);
    if min(size(arg1))==1, x = arg1(:)*ones(1,m); else x = arg1; end
    overlap = defaultoverlap; spacing = defaultspacing;
  else
    spacing = arg3; overlap = arg2;
    if any(size(overlap)~=[1 1]) | any(size(spacing)~=[1 1]),
      error('OVERLAP and SPACING must be scalars.');
    end
    if min(size(arg1))==1, y = arg1(:); else y = arg1; end
    [n,m] = size(y);
    x = [1:n]'*ones(1,m);
  end
elseif nargin==4, %bar(x,y,overlap,spacing) or bar(y,overlap,space,'kind')
  if isstr(arg4),
    kind = [lower(arg4) '  ']; % Protect against short string
    if all(kind(1:2)=='ov'), % overlap
      stacked = 0;
    elseif all(kind(1:2)=='st'), % stacked
      stacked = 1;
    else
      error('Last string argument must be ''stacked'' or ''overlap''.');
    end
    if min(size(arg1))==1, y = arg1(:); else y = arg1; end
    [n,m] = size(y);
    x = [1:n]'*ones(1,m);
    overlap = arg2; spacing = arg3;
  else
    if any(size(arg3)~=[1 1]) | any(size(arg4)~=[1 1]),
      error('OVERLAP and SPACING must be scalars.');
    end
    if min(size(arg2))==1, y = arg2(:); else y = arg2; end
    [n,m] = size(y);
    if min(size(arg1))==1, x = arg1(:)*ones(1,m); else x = arg1; end
    overlap = arg3; spacing = arg4;
  end
elseif nargin==5, % bar(x,y,overlap,spacing,'kind')
  if any(size(arg3)~=[1 1]) | any(size(arg4)~=[1 1]),
    error('OVERLAP and SPACING must be scalars.');
  end
  if ~isstr(arg5),
    error('Last string argument must be ''stacked'' or ''overlap''.');
  else
    kind = [lower(arg5) '  ']; % Protect against short string
    if all(kind(1:2)=='ov'), % overlap
      stacked = 0;
    elseif all(kind(1:2)=='st'), % stacked
      stacked = 1;
    else
      error('Last string argument must be ''stacked'' or ''overlap''.');
    end
  end
  if min(size(arg2))==1, y = arg2(:); else y = arg2; end
  [n,m] = size(y);
  if min(size(arg1))==1, x = arg1(:)*ones(1,m); else x = arg1; end
  overlap = arg3; spacing = arg4;
end

if nargin<3,
  overlap = defaultoverlap; spacing = defaultspacing;
end
scale = (1+overlap);

% Check sizes
if any(size(x)~=size(y)), error('X and Y must be the same size.'); end

if stacked & (m>1), y = cumsum(y')'; end

nn = 5*n;
yy = zeros(nn+1,m);
xx = yy;
yy(3:5:nn,:) = y;
yy(4:5:nn,:) = y;

notequal = max(abs(diff(diff(x)))) > max(max(abs(x)))*sqrt(eps);

if max(diff(x))==0, notequal=[]; end % Special case 

if isempty(notequal), % Scalar case and special case
    delta = 1;
    t = x - 0.5*delta;
    xx(1:5:nn,:) = t + delta*scale;
    xx(2:5:nn,:) = t + delta*scale;
    xx(3:5:nn,:) = t + delta*scale;
    xx(4:5:nn,:) = t + (1-scale)*delta;
    xx(5:5:nn,:) = t + (1-scale)*delta;
    xx(nn+1,:) = xx(nn,:);
elseif ~notequal
	if m==1 | stacked, spacing = 0; end
    if stacked,
      delta = ones(n,1) * (max(x) - min(x)) * (1-spacing) / (n-1);
      t = x - 0.5*delta;
    else
      delta = ones(n,1) * (max(x) - min(x)) * (1-spacing) / (n-1) / m ;
      t = x - 0.5*delta*m + (ones(n,1)*(0:m-1)).*delta;
    end
    xx(1:5:nn,:) = t;
    xx(2:5:nn,:) = t + delta*scale;
    xx(3:5:nn,:) = t + delta*scale;
    xx(4:5:nn,:) = t + (1-scale)*delta;
    xx(5:5:nn,:) = t + (1-scale)*delta;
    xx(nn+1,:) = xx(nn,:);
else    % spacing is unequal -- use smallest value
	if m==1 | stacked, spacing = -overlap; end
	if stacked,
    	dx = diff(x)/2*(1-spacing);
    	t = x;
	else
    	dx = diff(x)/2*(1-spacing)/m;
    	t = x + (ones(n,1)*(-(m-1)/2:(m-1)/2)).*[dx(1,:);dx]*2;
	end
    xx(1:5:nn+1,:) = [t;t(n,:)] - [dx(1,:);dx;dx(n-1,:)];
    xx(2:5:nn+1,:) = t - scale*[dx(1,:);dx];
    xx(3:5:nn+1,:) = xx(2:5:nn+1,:);
    xx(4:5:nn+1,:) = t + scale*[dx;dx(n-1,:)];
    xx(5:5:nn+1,:) = xx(4:5:nn+1,:);
    xx(nn,:) = t(n,:) + scale*dx(n-1,:);
    xx(nn+1,:) = xx(nn,:);
end

if nargout==2,
    xo = xx;
    yo = yy;
else
    cax = newplot;
    next = lower(get(cax,'NextPlot'));
    hold_state = ishold;
    co = get(gcf,'defaultaxescolororder'); n = size(co,1);
    edgec = get(gcf,'defaultaxesxcolor');
    h = []; 
    cc = ones(size(xx,1),1); cc(1:5:nn,:) = cc(1:5:nn,:);
    for i=size(xx,2):-1:1,
      h = [h,patch('xdata',xx(:,i),'ydata',yy(:,i),'cdata',i*cc, ...
             'FaceColor','flat','EdgeColor',edgec)];
    end
    set(cax,'Layer','Top','box','on')
    if all(all(floor(x)==x)) & (size(x,1)<16),  % Set ticks
      set(cax,'xtick',x(:,1))
    end
    if notequal, hold on, plot(x(:,1),zeros(size(x,1),1),'*'), end
    if ~hold_state, hold off, view(2), set(cax,'NextPlot',next); end
	if nargout==1, xo = h; end
end
